# -*- coding: utf-8 -*-
"""
Created on Sat Jul  7 13:40:19 2018

@author: david
"""

import random
import os
import numpy as np


def milvang_win_or_draw_prob(r1, r2):
  # This section is lifted from Milvang,
  # http://www.nordstrandsjakk.no/documents/spp/Probability.pdf
  
  rm = (r1 + r2)/2.0
  rd = r1 - r2
  rmq = max((rm - 1200.)/1200., 0.)
  
  wcl = 40.
  wcv = 0.45 - 0.1*rmq*rmq
  wll = -1492.0 + rm *0.391;
  wul = 1691.0 - rm *0.428;
  bcl = -80.0;
  bcv = 0.46 - 0.13 * rmq * rmq;
  bll = -1753 + rm * 0.416;
  bul = 1428 - rm * 0.388;
  wf1 = (rd - wll) / (wcl - wll);
  wf2 = (rd - wul) / (wcl - wul);
  bf1 = (rd - bll) / (bcl - bll);
  bf2 = (rd - bul) / (bcl - bul);
  
  if rd < wll:
    pw = 0.
  elif (wll <= rd) and (rd <= wcl):
    pw = wcv*wf1*wf1
  elif (wcl <= rd) and (rd <= wul):
    pw = 1.0 - (1.0 - wcv)*wf2*wf2
  elif (rd > wul):
    pw = 1.0
  
  if (rd < bll):
    pb = 1.0
  elif (bll <= rd) and (rd <= bcl):
    pb = 1.0 - (1.0 - bcv)*bf1*bf1
  elif (bll <= rd) and (rd <= bul):
    pb = bcv*bf2*bf2
  elif (rd > bul):
    pb = 0.0
  
  pd = 1.0 - pw - pb
  return [pw, pd + pw]


def append_xxc_accelerations(lines,
                             players,
                             i_round,
                             acceleration):
  num_players = len(players)
  virtual_points = acceleration(players, i_round)
  for i in range(num_players):
    lines[i] += "{:5.1f}".format(virtual_points[i])
  
  return lines


def write_trf(out_file, pre, players, scores, results, accel):
  with open(out_file, "w") as f:
    f.write(pre)
    
    for i in range(len(players)):
      f.write("{}{}{}\n".format(players[i],
                                scores[i],
                                results[i]))
    
    for line in accel:
      f.write("{}\n".format(line))
  
  return 0


def result_string(score):
  if score == 1:
    return "1"
  elif score == 0.5:
    return "="
  else:
    return "0"


def simulate_tournament(players,
                        acceleration = lambda p, i: [0]*len(p),
                        num_rounds = 6,
                        trf_file_base_name = "swiss_sim",
                        tournament_name = "sim",
                        city = "Perth",
                        federation = "AUS",
                        arbiter = "-",
                        top_seed_color = random.choice(["white", "black"]),
                        modelled_rating_floor = 700,
                        misrate_player = False):
  """players: list of dictionaries of the form
              [{"name": "Player 1's name", "rating": 2300},
               {"name": "Player 2's name", "rating": 2250},
               ... ]
              Can optionally have key "strength" to indicate a player's true
              strength, which can be different from the rating used for
              pairings.
     
     acceleration: function that takes arguments of players and round number,
                   and returns a list of virtual points to assign.
  
  """
  num_players = len(players)
  
  for i in range(num_players):
    players[i]["score"] = 0
    players[i]["prog_scores"] = [0]
    players[i]["games"] = []
    if "strength" not in players[i]:
      players[i]["strength"] = players[i]["rating"]
    
    if misrate_player:
      players[num_players - 5]["strength"] = 2000
      players[num_players - 4]["strength"] = 1800
  
  trf_preamble_lines = """012 {}
022 {}
032 {}
102 {}
XXR {}
XXC {}1\n""".format(tournament_name,
                    city,
                    federation,
                    arbiter,
                    num_rounds,
                    top_seed_color)
  
  
  trf_player_lines = ["001" for i in range(num_players)]
  trf_player_scores =  ["" for i in range(num_players)]
  trf_player_results = ["" for i in range(num_players)]
  trf_accel_lines = ["XXA" for i in range(num_players)]
  
  for i in range(num_players):
    trf_player_lines[i] += "{0:>5}      ".format(i+1)
    trf_player_lines[i] += "{0:<33} " .format(players[i]["name"])
    trf_player_lines[i] += "{0:>4}" .format(players[i]["rating"])
    trf_player_lines[i] += "  WA           0 0000/00/00"
    trf_player_scores[i] = "  0.0{0:>5}".format(i+1)
    trf_accel_lines[i] += "{0:>5}".format(i+1)
  
  
  trf_accel_lines = append_xxc_accelerations(trf_accel_lines,
                                             players,
                                             1,
                                             acceleration)
  
  output_file = "{}_0.trfx".format(trf_file_base_name)
  pairings_file = "{}_0_pairs.txt".format(trf_file_base_name)
  
  write_trf(output_file,
            trf_preamble_lines,
            trf_player_lines,
            trf_player_scores,
            trf_player_results,
            trf_accel_lines)
  
  
  # ~~~~~ Now to simulate the tournament itself ~~~~~
  
  for i_round in range(num_rounds):
    os.system("./bbpPairings --dutch {} -p {}".format(output_file,
                                                      pairings_file))
    
    with open(pairings_file) as f:
      pairs = f.readlines()
    
    for i in range(num_players):
      players[i]["games"].append({"opponent": -1,
                                  "color": "",
                                  "result": -1,
                                  "expected_score": -1})
    
    for (i, e) in enumerate(pairs):
      if i > 0:
        cells = e.strip().split(" ")
        
        player1 = int(cells[0]) - 1
        player2 = int(cells[1]) - 1
        
        rating1 = players[player1]["strength"]
        rating2 = players[player2]["strength"]
        
        rating1 = max(rating1, modelled_rating_floor)
        rating2 = max(rating2, modelled_rating_floor)
        
        players[player1]["games"][i_round]["opponent"] = player2
        players[player1]["games"][i_round]["color"] = "w"
        
        players[player2]["games"][i_round]["opponent"] = player1
        players[player2]["games"][i_round]["color"] = "b"
        
        probs = milvang_win_or_draw_prob(rating1, rating2)
        x = random.random()
        if x < probs[0]:
          score1 = 1.
          score2 = 0.
        elif x < probs[1]:
          score1 = 0.5
          score2 = 0.5
        else:
          score1 = 0.
          score2 = 1.
        
        players[player1]["games"][i_round]["result"] = score1
        players[player2]["games"][i_round]["result"] = score2
        
        players[player1]["score"] += score1
        players[player2]["score"] += score2
        
        players[player1]["prog_scores"].append(players[player1]["score"])
        players[player2]["prog_scores"].append(players[player2]["score"])
        
        players[player1]["games"][i_round]["expected_score"] = 0.5*(probs[0] + probs[1])
        players[player2]["games"][i_round]["expected_score"] = 1.0 - 0.5*(probs[0] + probs[1])
        
    for i in range(num_players):
      if players[i]["games"][i_round]["opponent"] < 0:
        # Player was unpaired, full-point bye.
        players[i]["score"] += 1.
    
    # Need to rank the players....
    scores = [-p["score"] for p in players]
    sorted_i = list(np.argsort(scores))
    ranks = [sorted_i.index(i) for i in range(num_players)]
    
    scores = [-s for s in scores]
    
    
    for i in range(num_players):
      trf_player_scores[i] = "{0:5.1f}{1:>5}".format(scores[i], ranks[i]+1)
      
      if players[i]["games"][i_round]["opponent"] < 0:
        trf_player_results[i] += "  0000 - U"
      else:
        trf_player_results[i] += " {0:>5} {1} {2}".format(
          players[i]["games"][i_round]["opponent"] + 1,
          players[i]["games"][i_round]["color"],
          result_string(players[i]["games"][i_round]["result"]))
      
    trf_accel_lines = append_xxc_accelerations(trf_accel_lines,
                                               players,
                                               i_round + 2,
                                               acceleration)
    
    output_file = "{}_{}.trfx".format(trf_file_base_name, i_round+1)
    pairings_file = "{}_{}_pairs.txt".format(trf_file_base_name, i_round+1)  
    
    write_trf(output_file,
              trf_preamble_lines,
              trf_player_lines,
              trf_player_scores,
              trf_player_results,
              trf_accel_lines)
    
  return players
